<?php

namespace Modules\Contacts\Imports;

use Maatwebsite\Excel\Concerns\ToModel;
use Maatwebsite\Excel\Concerns\WithHeadingRow;
use Maatwebsite\Excel\Concerns\WithChunkReading;
use Modules\Contacts\Models\Contact;
use Modules\Contacts\Models\Field;

class ContactsImport implements ToModel, WithHeadingRow, WithChunkReading
{
    protected $group_id;

    public function __construct($group_id = null)
    {
        $this->group_id = $group_id;
    }

    public function chunkSize(): int
    {
        return 150;
    }

    public function model(array $row)
    {
        $keys = array_keys($row);
        $keysForFields = [];

        // Create missing custom fields if needed
        foreach ($keys as $key => $value) {
            $keysForFields[$key] = $this->getOrMakeField($value);
        }

        // Normalize phone format
        $phone = strpos($row['phone'], '+') !== false ? $row['phone'] : '+' . $row['phone'];

        // Check for existing contact by phone
        $contact = Contact::where('phone', $phone)->first();

        if (!$contact) {
            // Create new contact if not found
            $contact = Contact::create([
                'name'  => $row['name'] ?? '',
                'phone' => $phone,
            ]);
        }

        // Optional avatar field
        if (!empty($row['avatar'])) {
            $contact->avatar = $row['avatar'];
        }

        // Attach dynamic field values
        foreach ($keysForFields as $key => $fieldID) {
            if ($fieldID != 0 && !empty($row[$keys[$key]])) {
                $contact->fields()->syncWithoutDetaching([
                    $fieldID => ['value' => $row[$keys[$key]]],
                ]);
            }
        }

        // ✅ Attach contact to the specified group (without overwriting existing groups)
        if ($this->group_id) {
            $contact->groups()->syncWithoutDetaching([$this->group_id]);
        }

        $contact->save();

        return $contact;
    }

    private function getOrMakeField($field_name)
    {
        // Skip default columns
        if (in_array($field_name, ['name', 'phone', 'avatar'])) {
            return 0;
        }

        $field = Field::where('name', $field_name)->first();

        if (!$field) {
            $field = Field::create([
                'name' => $field_name,
                'type' => 'text',
            ]);
        }

        return $field->id;
    }
}